//------------------------------------------------------------------------------
// File: BadSeed.cs
// Controls the bad seed task and its vertex methods
// Author: Matthew Rudge
//------------------------------------------------------------------------------

$BadSeed::PayoffAmount = 1200;

//-Bridge Defense Helper Functions----------------------------------------------
////////////////////////////////////////////////////////////////////////////////
//! Starts the tasks for and initializes the bad seed mission
////////////////////////////////////////////////////////////////////////////////
function startBadSeedTasks()
{
   InitializeQuestGui();
   tsStartTask("BadSeed", tsGetStartingVertexes("BadSeed"));
}

////////////////////////////////////////////////////////////////////////////////
//! Cleans up any left-over tasks or changes from the bad seed mission
////////////////////////////////////////////////////////////////////////////////
function endBadSeedTasks()
{
   tsEndTask("BadSeed");
}

////////////////////////////////////////////////////////////////////////////////
//! Reset Badseed Task
////////////////////////////////////////////////////////////////////////////////
function resetBadSeedTask()
{
   TaskModifier.strResetVertex("MainmapTasks", "BadSeedMapTaskAccept");
   TaskModifier.strMarkVertex("MainmapTasks", "BadSeedMapTaskAccept", $TSV_AVAIL);
}

////////////////////////////////////////////////////////////////////////////////
//! Rewards the player for mission completion and cleans the mission tasks upon
//! exit
////////////////////////////////////////////////////////////////////////////////
function winBadSeedTasks()
{
   tsModifyResource($Resource::Experience, 25, true);
   SendProductionToClient(HeroCharacter, "25 experience");
   badseedgal.deleteObject();
   //Quest log
   tsCompleteTaskInTracker("Bad Seed", "Help Sadey Malla and her town.");
   
   TaskModifier.strMarkVertex("MainmapTasks", "BadSeedComplete", $TSV_AVAIL);
}
//------------------------------------------------------------------------------

//-BadSeedIntro Vertex----------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is responsible for the opening dialog of the task
////////////////////////////////////////////////////////////////////////////////
function BadSeedIntro::onEnter(%this)
{
   Parent::onEnter(%this);
   
   tsSendTaskPopUpDlg("ID_TASKNAME_BADSEED", "id_badseed_introstart", $TaskButton::Okay, "civ1", "id_badseed_introstart");
   
   // Spawn farmer (somewhere)
   %obj = tsCreateObj(MaleCiv, "badfarmer", $OST_NEUTRAL, "0 0", "badfarmer");
   %obj.name = "Evan Jasper";
}

function BadSeedIntro::onMessage(%this, %msg, %params)
{
   %firstComplete = %this.isFunctionComplete($TSF_DIALOGDONE, 0);
   Parent::onMessage(%this, %msg, %params);
   
   // If we have finished the first dialog, proceed to the next one
   if(!%firstComplete && %this.isFunctionComplete($TSF_DIALOGDONE, 0)) {
      tsSendTaskPopUpDlg("ID_TASKNAME_BADSEED", "id_badseed_introend", $TaskButton::Okay, "HeroCharacter", "id_badseed_introend");
   }
}
//------------------------------------------------------------------------------

//-BadSeedGoldMineCheck Vertex------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is responsible for monitoring the status of the gold mine. Player
// loses scenario if gold mine changes team or is destroyed
////////////////////////////////////////////////////////////////////////////////
function BadSeedGoldMineCheck::onUpdate(%this)
{
   // If gold mine is destroyed, the player loses
   if(!isObject("BSGoldmine")) {
      tsRestartScenario();
      return;
   }
   
   // If gold mine has changed teams, the player loses
   if(!BSGoldmine.isOnTeam($OST_PLAYER)) {
      TaskModifier.markVertex(%this.getTaskId(), %this.getVertexId(), $TSV_CMPLETE);
      return;
   }
}
//------------------------------------------------------------------------------

//-BadSeedGoldMineSold Vertex---------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is responsible for displaying the lose dialog after your gold
// mine has been bought by the farmer
////////////////////////////////////////////////////////////////////////////////
function BadSeedGoldMineSold::onExit(%this)
{
   Parent::onExit(%this);
   tsRestartScenario();
}
//------------------------------------------------------------------------------

//-BadSeedBuyTimer Vertex-------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is responsible for buying a building from the player after a set
// amount of time has passed
////////////////////////////////////////////////////////////////////////////////
function BadSeedBuyTimer::onExit(%this)
{
   // Create a list of buildings that we were randomly select from to convert
   // to team neutral
   %teamBldgs      = new SLGameObjList();
   %potentialBldgs = new SLGameObjList();
   
   %allBldgs = slgGetServerBuildingList();
   %allBldgs.getTeamList($OST_PLAYER, %teamBldgs);
   
   // Loop through team buildings and find non-luxury buildings
   %count = %teamBldgs.getSize();
   for(%i = 0; %i < %count; %i++) {
      %bldg = %teamBldgs.getID(%i);
      
      // Don't include buildings that are not in production state
      %cmpBldg = slgQueryInterface(%bldg, $CID_BUILDING);
      if(%cmpBldg.getState() != $BuildingState::Production) {
         continue;
      }
      // Only include town buildings
      if(!%bldg.isTownBuilding()) {
         continue;
      }
      // Don't include starting gold mine
      if(%bldg.getName() $= "BSGoldmine") {
         continue;
      }

      %potentialBldgs.addObject(%bldg);
   }
   
   // Choose random object
   if(%potentialBldgs.getSize() > 0) {
      %target = %potentialBldgs.getID(getRandom(0, %potentialBldgs.getSize() - 1));
   }
   // Target is gold mine
   else {
      %target = BSGoldmine.getId();
   }
   
   // Convert team of target
   tsConvertObjTeam(%target, $OST_NEUTRAL);
   tsMoveCameraToObject(%target, false);
   
   // Send alert message
   %msg = tsGetTaskString("ID_BADSEED_FARMERBUY");
   %msg = slgFormatUIString(%msg, %target.name);
   tsSendAlertMessage(%msg);
   
   // Clean-up
   %teamBldgs.delete();
   %potentialBldgs.delete();

}
function BadSeedBuyTownDialog::onExit()
{
   // Get Random town civ 
   %teamList = tsCreateObjList();
   %typeList = tsCreateObjList();
   %objList = slgGetServerCharacterList();
   %objList.getTeamList($OST_PLAYER, %teamList);
   %teamList.getTypeList("MaleCiv", %typeList);
   if(%typeList.getSize() <= 0)
   {
      %teamList.getTypeList("FemaleCiv", %typeList);
   }
   if(%typeList.getSize() <= 0)
   {
      // Use Hero
      %townciv = "herocharacter";
   }
   else
   {
      %townciv = %typeList.getRandomObject();
   }
   // Random town civ dialog after town building lost
   tsSendTaskPopUpDlg("ID_TASKNAME_BADSEED", "id_badseed_bldglossdialog", $TaskButton::Okay, %townciv);
   tsDestroyObjList(%teamList);
   tsDestroyObjList(%typeList);
   BadseedBuyReset();
}
function BadseedBuyReset()
{
   // Reset the parent vertex
   TaskModifier.strResetVertex("BadSeed", "BadSeedBuyTimerDialog");
   TaskModifier.strMarkVertex("BadSeed", "BadSeedBuyTimerDialog", $TSV_AVAIL);
}
//------------------------------------------------------------------------------

//-BadSeedBuyTimerDialog Vertex-------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is responsible for displaying the "I've taken your building" 
// dialog after a set amount of time has passed
////////////////////////////////////////////////////////////////////////////////
function BadSeedBuyTimerDialog::onExit(%this)
{
   // Display dialog of building lost
   tsSendTaskPopUpDlg("ID_TASKNAME_BADSEED", "id_badseed_bldgloss", $TaskButton::Okay, "badfarmer", "id_badseed_bldgloss");
}
//------------------------------------------------------------------------------

//-BadSeedGoldCheck Vertex------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is responsible for checking to see if the player's gold 
// requirement has been met
////////////////////////////////////////////////////////////////////////////////
function BadSeedGoldCheck::onEnter(%this)
{
   Parent::onEnter(%this);
   
   // Switch neutral team units to player team
   tsConvertObjTeam("BSGoldmine", $OST_PLAYER);
   tsConvertObjTeam("BSCabin", $OST_PLAYER);
   tsConvertObjTeam("BSShack", $OST_PLAYER);
   tsConvertObjTeam("BSFarm", $OST_PLAYER);
   tsConvertObjTeam("BSwell", $OST_PLAYER);
   tsConvertObjTeam("civ1", $OST_PLAYER);
   tsConvertObjTeam("civ2", $OST_PLAYER);
   tsConvertObjTeam("civ3", $OST_PLAYER);
   tsSkinObj("civ1", "base");
   tsSkinObj("civ2", "base");
   tsSkinObj("civ3", "base");
   
   // Add quest logs
   tsAddQuestToTracker(true, "Bad Seed", "", "Save up Gold to buy the deeds to the town back.", "Reward: 25 Experience");
   tsReadyQuest("Bad Seed");
   tsAddTaskToTracker("Bad Seed", "Collect "@$BadSeed::PayoffAmount@" Gold.", "Collect "@$BadSeed::PayoffAmount@" ounces of Gold to buy the town's deeds back.");
   tsAddTaskToTracker("Bad Seed", "Protect your Gold Mine.", "Don't lose your Gold Mine.");
}

function BadSeedGoldCheck::onUpdate(%this)
{
   // Complete this vertex if gold amount has been reached
   %count = rsGetResourceCount($Resource::Gold);
   if(%count >= $BadSeed::PayoffAmount) 
   {
      tsCompleteVertexes("BadSeed", "BadSeedGoldCheck");
      TaskModifier.strMarkVertex("BadSeed", "BadSeedBuyTimerDialog", $TSV_CLOSED);
      TaskModifier.strMarkVertex("BadSeed", "BadSeedGoldCheck", $TSV_CMPLETE);
   }
}
//------------------------------------------------------------------------------

//-BadSeedGoldCheckDialog Vertex------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is responsible for displaying and resolving the dialog between
// the farmer and the hero
////////////////////////////////////////////////////////////////////////////////
function BadSeedGoldCheckDialog::onEnter(%this)
{
   Parent::onEnter(%this);
   
   // Try to use a civilian first, then the hero if necessary
   %team  = new SLGameObjList();
   %chars = new SLGameObjList();
   %base  = slgGetServerCharacterList();
   %base.getTeamList($OST_PLAYER, %team);
   %team.getTypeList("FemaleCiv", %chars);
   if(%chars.getSize() == 0) {
      %team.getTypeList("MaleCiv", %chars);
   }
   if(%chars.getSize() != 0) {
      %speaker = %chars.getID(0);
   }
   else {
      %speaker = HeroCharacter.getId();
   }
   %team.delete();
   %chars.delete();
   
   // Spawn dialog between farmer, civilian, and hero
   tsSendTaskPopUpDlg("ID_TASKNAME_BADSEED", "id_badseed_payoffstart", $TaskButton::Okay, %speaker);
   tsSendTaskPopUpDlg("ID_TASKNAME_BADSEED", "id_badseed_payoffmid", $TaskButton::Okay, "badfarmer");
   tsSendTaskPopUpDlg("ID_TASKNAME_BADSEED", "id_badseed_payoffend", $TaskButton::Okay, "HeroCharacter", "id_badseed_payoffend");
}

function BadSeedGoldCheckDialog::onExit(%this)
{
   tsAddTaskToTracker("Bad Seed", "Guard your Gold Mine.", "Hire some guns to protect your Gold Mine at all costs.");
   tsCompleteTaskInTracker("Bad Seed", "Collect "@$BadSeed::PayoffAmount@" Gold.");
   tsCompleteTaskInTracker("Bad Seed", "Protect your Gold Mine.");
}
//------------------------------------------------------------------------------

function BadSeedBanditAttack::onEnter(%this)
{
   %male = tsCreateObj("Malebandit", "farmhand1", $OST_NEUTRAL, "-464 -199", "farmhand1");
   %female = tsCreateObj("Femalebandit", "farmhand2", $OST_NEUTRAL, "-461 -192", "farmhand2");
   
   %male.name = "Cletus Jasper";
   %female.name = "Annie Jasper";
}

//-BadSeedBanditAttack Vertex---------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is responsible for spawning the bandit attack after the player 
// has collected the required gold, progressed through the dialog, and waited
// for a set amount of time
////////////////////////////////////////////////////////////////////////////////
function BadSeedBanditAttack::onExit(%this)
{
   %male = nameToId("farmhand1");
   %female = nameToId("farmhand2");
   
   %male.setTeam($OST_BANDIT);
   %female.setTeam($OST_BANDIT);
   
   // Attack target
   tsStartBanditAttack(%male, "BSGoldmine");
   tsStartBanditAttack(%female, "BSGoldmine");
}
//------------------------------------------------------------------------------

//-BadSeedBanditDefeat Vertex---------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is responsible for spawning the banker after the bandit attack
// has been defeated
////////////////////////////////////////////////////////////////////////////////
function BadSeedBanditDefeat::onExit(%this)
{
   %banker = tsCreateObj("Maleciv", "bankerdude", $OST_NEUTRAL, "-464 -199", "bankerdude");
   %banker.name = "Archibald Abercrombie";
   tsSkinObj(%banker,"default");
   tsMoveObjToTarget(%banker, BSGoldmine.getId());
   tsCompleteTaskInTracker("Bad Seed", "Protect your Gold Mine.");
}
//------------------------------------------------------------------------------

//-BadSeedBankerArrives Vertex--------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is responsible for creating the ending dialog with the banker 
// after the banker has been created
////////////////////////////////////////////////////////////////////////////////
function BadSeedBankerArrives::onExit(%this)
{
   %dlgData = tsCreateTaskDlgData("ID_TASKNAME_BADSEED", true, $TaskButton::Accept, "id_badseed_denoument");
   tsAddTaskDialog("id_badseed_denoument", %dlgData);
   tsSetCancelButtonText("ID_CANCEL_NOTYET", %dlgData);
   tsSetAcceptButtonText("ID_ACCEPT_PAY", %dlgData);
   tsSendTaskDlgMsg(%dlgData, "bankerdude");
   tsDestroyTaskDlgData(%dlgData);
}
//------------------------------------------------------------------------------

//-BadSeedWin Vertex------------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is responsible for moving the player back to the main map after
// the player clicks accept on the banker's dialog
////////////////////////////////////////////////////////////////////////////////
function BadSeedWin::onExit(%this)
{
   %scenario = new SLScenario() {
      stack = false;
   };
   LoadScenario(%scenario, endBadSeedTasks, winBadSeedTasks);
   
   // complete the bad seed satellite
   mapsatellite_badseed.disabled = false;
}
//------------------------------------------------------------------------------
// End BadSeed.cs